<?php

namespace Tests\Feature;

use App\Models\App;
use App\Services\ZipArchiveService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;
use ZipArchive;

class ZipArchiveServiceTest extends TestCase
{
    use RefreshDatabase;

    private ZipArchiveService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = new ZipArchiveService();
        Storage::fake('public');
        Storage::fake('local');
    }

    /** @test */
    public function it_creates_zip_archive_with_app_info_files()
    {
        // Arrange
        $apps = App::factory()->count(3)->create([
            'apk_path' => null,
            'download_url' => null,
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'Test Collection',
            $apps->pluck('id')->toArray(),
            false // Don't include APK files
        );

        // Assert
        $this->assertFileExists($archivePath);
        
        $zip = new ZipArchive();
        $zip->open($archivePath);
        
        $this->assertNotFalse($zip->locateName('README.txt'));
        
        foreach ($apps as $app) {
            $safePackage = \Illuminate\Support\Str::slug($app->package_name);
            $this->assertNotFalse($zip->locateName("${safePackage}/info.txt"));
        }
        
        $zip->close();
    }

    /** @test */
    public function it_includes_apk_from_local_storage()
    {
        // Arrange
        $apkContent = str_repeat('APK_CONTENT', 1000);
        Storage::disk('local')->put('apks/test-app.apk', $apkContent);
        
        $app = App::factory()->create([
            'package_name' => 'com.test.app',
            'apk_path' => 'apks/test-app.apk',
            'download_url' => null,
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'APK Test',
            [$app->id],
            true
        );

        // Assert
        $zip = new ZipArchive();
        $zip->open($archivePath);
        
        $apkFileName = 'com-test-app/com-test-app.apk';
        $this->assertNotFalse($zip->locateName($apkFileName));
        $this->assertEquals($apkContent, $zip->getFromName($apkFileName));
        
        $zip->close();
    }

    /** @test */
    public function it_downloads_apk_from_url_when_local_not_available()
    {
        // Arrange
        $apkContent = str_repeat('DOWNLOADED_APK', 1000);
        
        Http::fake([
            'https://example.com/app.apk' => Http::response($apkContent, 200),
        ]);
        
        $app = App::factory()->create([
            'package_name' => 'com.download.app',
            'apk_path' => null,
            'download_url' => 'https://example.com/app.apk',
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'Download Test',
            [$app->id],
            true
        );

        // Assert
        $zip = new ZipArchive();
        $zip->open($archivePath);
        
        $apkFileName = 'com-download-app/com-download-app.apk';
        $this->assertNotFalse($zip->locateName($apkFileName));
        
        $zip->close();
        
        Http::assertSent(function ($request) {
            return $request->url() === 'https://example.com/app.apk';
        });
    }

    /** @test */
    public function it_records_error_when_apk_file_missing()
    {
        // Arrange
        $app = App::factory()->create([
            'package_name' => 'com.missing.app',
            'apk_path' => 'non-existent/path.apk',
            'download_url' => null,
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'Missing APK Test',
            [$app->id],
            true
        );

        // Assert
        $errors = $this->service->getErrors();
        $this->assertNotEmpty($errors);
        $this->assertStringContainsString('com.missing.app', $errors[0]);
    }

    /** @test */
    public function it_records_error_when_download_fails()
    {
        // Arrange
        Http::fake([
            'https://example.com/fail.apk' => Http::response('Not Found', 404),
        ]);
        
        $app = App::factory()->create([
            'package_name' => 'com.fail.app',
            'apk_path' => null,
            'download_url' => 'https://example.com/fail.apk',
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'Failed Download Test',
            [$app->id],
            true
        );

        // Assert
        $errors = $this->service->getErrors();
        $this->assertNotEmpty($errors);
        $this->assertStringContainsString('404', $errors[0]);
    }

    /** @test */
    public function it_throws_exception_when_no_apps_found()
    {
        // Assert
        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage('No apps found');

        // Act
        $this->service->createFromAppCollection('Empty', [999999]);
    }

    /** @test */
    public function it_creates_errors_file_when_issues_occur()
    {
        // Arrange
        $app = App::factory()->create([
            'package_name' => 'com.error.app',
            'apk_path' => 'missing.apk',
            'download_url' => null,
        ]);

        // Act
        $archivePath = $this->service->createFromAppCollection(
            'Error Log Test',
            [$app->id],
            true
        );

        // Assert
        $zip = new ZipArchive();
        $zip->open($archivePath);
        
        $this->assertNotFalse($zip->locateName('ERRORS.txt'));
        
        $errorsContent = $zip->getFromName('ERRORS.txt');
        $this->assertStringContainsString('com.error.app', $errorsContent);
        
        $zip->close();
    }

    /** @test */
    public function it_cleans_up_old_archives()
    {
        // Arrange
        Storage::disk('public')->put('exports/old-archive.zip', 'content');
        
        // Simulate old file by touching it (in real scenario)
        // For this test, we'll just verify the method runs
        
        // Act
        $deleted = $this->service->cleanupOldArchives(30);
        
        // Assert
        $this->assertIsInt($deleted);
    }

    /** @test */
    public function it_lists_available_archives()
    {
        // Arrange
        Storage::disk('public')->put('exports/test1.zip', 'content1');
        Storage::disk('public')->put('exports/test2.zip', 'content2');
        Storage::disk('public')->put('exports/not-a-zip.txt', 'text');

        // Act
        $archives = $this->service->listArchives();

        // Assert
        $this->assertCount(2, $archives);
        $this->assertEquals('test2.zip', $archives[0]['name']);
    }
}
