#!/usr/bin/env python3
"""
scripts/optimize_images.py

Simple image optimization helper using Pillow.

Usage:
  python3 scripts/optimize_images.py --dry-run
  python3 scripts/optimize_images.py --output public/images-optimized

The script will scan `public/images` and `resources/images` (if present), report potential savings,
and optionally write optimized files to the provided output directory (keeps originals untouched).

Requires: Pillow
  python3 -m pip install --user pillow

This script is safe to run in CI or on your host. It does not overwrite originals by default.
"""

import argparse
import os
import sys
from pathlib import Path
from io import BytesIO

try:
    from PIL import Image
except Exception:
    print("Pillow is required. Install with: python3 -m pip install --user pillow", file=sys.stderr)
    sys.exit(2)


IMAGE_DIRS = ["public/images", "resources/images"]
SUPPORTED_EXT = (".jpg", ".jpeg", ".png", ".webp")


def gather_images(root_dirs):
    images = []
    for d in root_dirs:
        p = Path(d)
        if not p.exists():
            continue
        for fp in p.rglob("*"):
            if fp.suffix.lower() in SUPPORTED_EXT and fp.is_file():
                images.append(fp)
    return images


def optimize_image(fp: Path, quality=85, webp_quality=80):
    """Return optimized bytes and format"""
    img = Image.open(fp)
    img_format = img.format or fp.suffix.replace('.', '').upper()
    out = BytesIO()

    if img.mode in ("P", "RGBA") and img_format in ("PNG",):
        # PNG flatten RGBA to RGB over white for web display optionally; but we'll save PNG optimized
        img.save(out, format="PNG", optimize=True)
        fmt = "PNG"
    elif img_format in ("JPEG", "JPG"):
        img = img.convert("RGB")
        img.save(out, format="JPEG", quality=quality, optimize=True)
        fmt = "JPEG"
    else:
        # For other (WEBP etc), try saving as WEBP
        img.save(out, format="WEBP", quality=webp_quality, method=6)
        fmt = "WEBP"

    out.seek(0)
    return out.read(), fmt


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--dry-run", action="store_true", help="Don't write files; only estimate savings")
    parser.add_argument("--output", default="public/images-optimized", help="Output directory for optimized images")
    parser.add_argument("--inplace", action="store_true", help="Overwrite originals with optimized outputs (creates .bak backups if none exist)")
    parser.add_argument("--quality", type=int, default=85, help="JPEG quality (0-100)")
    parser.add_argument("--webp-quality", type=int, default=80, help="WEBP quality (0-100)")
    args = parser.parse_args()

    images = gather_images(IMAGE_DIRS)
    if not images:
        print("No images found in: {}".format(', '.join(IMAGE_DIRS)))
        return

    total_in = 0
    total_out = 0
    tasks = []

    for img in images:
        try:
            in_size = img.stat().st_size
            total_in += in_size
            opt_bytes, fmt = optimize_image(img, quality=args.quality, webp_quality=args.webp_quality)
            out_size = len(opt_bytes)
            total_out += out_size
            rel = img.relative_to(Path.cwd()) if img.is_absolute() else img
            tasks.append((str(rel), in_size, out_size, fmt, opt_bytes))
        except Exception as e:
            print(f"Skipping {img}: {e}")

    print(f"Found {len(tasks)} image(s). Total input: {total_in / 1024:.1f} KB")

    # Show top savings
    tasks_sorted = sorted(tasks, key=lambda t: (t[1]-t[2]), reverse=True)
    for name, in_size, out_size, fmt, _ in tasks_sorted[:50]:
        saved = in_size - out_size
        pct = (saved / in_size * 100) if in_size else 0
        print(f"{name}: {in_size/1024:.1f}KB -> {out_size/1024:.1f}KB (saved {saved/1024:.1f}KB, {pct:.0f}%) as {fmt}")

    print(f"Estimated total after optimization: {total_out/1024:.1f} KB (saved {(total_in-total_out)/1024:.1f} KB)")

    if args.dry_run:
        print("Dry-run: no files written. To write optimized files, re-run without --dry-run and optionally set --output")
        return
    out_base = Path(args.output)
    if args.inplace:
        for name, in_size, out_size, fmt, opt_bytes in tasks:
            src = Path(name)
            # create a simple backup if not present
            backup = src.with_name(src.name + '.bak')
            try:
                if not backup.exists():
                    src.replace(backup)
                    # write optimized bytes to original path
                    with open(src, 'wb') as f:
                        f.write(opt_bytes)
                    # restore backup name (we want .bak to remain) - move backup back
                    backup_path = src.with_name(src.name + '.bak')
                    if not backup_path.exists():
                        # move temp backup to final name
                        backup.replace(backup_path)
                else:
                    # backup exists; overwrite original directly
                    with open(src, 'wb') as f:
                        f.write(opt_bytes)
            except Exception:
                # fallback: attempt safe write via temp
                try:
                    tmp = src.with_name(src.name + '.tmp')
                    with open(tmp, 'wb') as f:
                        f.write(opt_bytes)
                    tmp.replace(src)
                except Exception as e:
                    print(f"Failed to write optimized file for {src}: {e}")
        print("In-place optimization completed.")
    else:
        for name, in_size, out_size, fmt, opt_bytes in tasks:
            src = Path(name)
            # compute target path
            try:
                rel = src.relative_to('public')
                target = out_base / rel
            except Exception:
                # fallback preserve full relative structure
                target = out_base / src

            target.parent.mkdir(parents=True, exist_ok=True)
            with open(target, 'wb') as f:
                f.write(opt_bytes)

        print(f"Wrote optimized images to: {out_base}")


if __name__ == '__main__':
    main()
