#!/usr/bin/env python3
"""
Create a deployment ZIP excluding development files.
"""
import os
import zipfile
from datetime import datetime

ROOT = os.path.dirname(os.path.dirname(__file__))
EXCLUDE = [
    '.git', '.github', 'node_modules', '.venv', 'tests', 'scripts', '.vscode', '.idea',
    'vendor/bin', 'storage/framework/cache', 'storage/logs', '.env', '.env.local'
]

def should_exclude(path):
    for ex in EXCLUDE:
        if path.startswith(ex) or ('/' + ex + '/') in path:
            return True
    return False

def main():
    today = datetime.utcnow().strftime('%Y%m%d')
    out_name = f"apks.center-deploy-{today}.zip"
    out_path = os.path.join(ROOT, out_name)

    with zipfile.ZipFile(out_path, 'w', compression=zipfile.ZIP_DEFLATED) as zf:
        for dirpath, dirnames, filenames in os.walk(ROOT):
            # Compute archive root relative path
            rel_dir = os.path.relpath(dirpath, ROOT)
            if rel_dir == '.':
                rel_dir = ''

            # Skip excluded directories
            if should_exclude(rel_dir):
                continue

            for fn in filenames:
                rel_file = os.path.join(rel_dir, fn) if rel_dir else fn
                if should_exclude(rel_file):
                    continue
                # Skip the output file itself
                if os.path.abspath(os.path.join(ROOT, rel_file)) == os.path.abspath(out_path):
                    continue
                abs_file = os.path.join(dirpath, fn)
                try:
                    zf.write(abs_file, rel_file)
                except Exception:
                    pass

    print('Created:', out_path)

if __name__ == '__main__':
    main()
