<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Http\Request;
use App\Http\Controllers\AppController;
use App\Http\Controllers\DownloadController;

/* -------------------------------------------------------------------------- */
/* WEB ROUTES                                                                 */
/* -------------------------------------------------------------------------- */
Route::middleware('web')->group(function () {
    Route::get('/', [AppController::class, 'index'])->name('home');

    // Games
    Route::get('/games', [AppController::class, 'games'])->name('apps.games');

    // Apps routes
    Route::prefix('apps')->group(function () {
        Route::get('latest', [AppController::class, 'latest'])->name('apps.latest');
        Route::get('popular', [AppController::class, 'popular'])->name('apps.popular');
        Route::get('games', [AppController::class, 'games'])->name('apps.games');
        Route::get('{app:slug}', [AppController::class, 'show'])->name('apps.show');
    });

    // Categories
    Route::get('/categories', [AppController::class, 'categories'])->name('apps.categories');
    Route::get('/categories/{category:slug}', [AppController::class, 'category'])->name('categories.show');

    // Search
    Route::get('/search', [AppController::class, 'search'])->name('apps.search');

    // Download
    Route::prefix('download')->group(function () {
        Route::get('{app:slug}/wait', [DownloadController::class, 'show'])->name('download.show');
        Route::get('{app:slug}/process', [DownloadController::class, 'process'])->name('download.process');
    });
});

/* -------------------------------------------------------------------------- */
/* API ROUTES - Version 1                                                     */
/* -------------------------------------------------------------------------- */
Route::prefix('v1')->middleware(['throttle:60,1'])->group(function () {
    /* =============================== Apps ================================= */
    Route::prefix('apps')->group(function () {
        Route::get('latest', [AppController::class, 'apiLatest']);
        Route::get('category/{category:slug}', [AppController::class, 'apiByCategory']);
        Route::get('{package_name}', [AppController::class, 'apiShow']);

        // Search endpoint
        Route::get('search', function (Request $request) {
            $query = $request->get('q');

            if (empty($query)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Search query is required',
                ], 400);
            }

            $apps = \App\Models\App::active()
                ->where(function ($q) use ($query) {
                    $q->where('name', 'LIKE', "%{$query}%")
                      ->orWhere('description', 'LIKE', "%{$query}%")
                      ->orWhere('package_name', 'LIKE', "%{$query}%");
                })
                ->with('category')
                ->paginate($request->get('per_page', 20));

            return response()->json([
                'success' => true,
                'data' => $apps->items(),
                'pagination' => [
                    'current_page' => $apps->currentPage(),
                    'last_page' => $apps->lastPage(),
                    'per_page' => $apps->perPage(),
                    'total' => $apps->total(),
                    'next_page_url' => $apps->nextPageUrl(),
                    'prev_page_url' => $apps->previousPageUrl(),
                ],
            ]);
        });
    });

    /* ============================ Categories ============================== */
    Route::get('categories', function () {
        $categories = \App\Models\Category::active()
            ->ordered()
            ->withCount(['apps as apps_count' => fn ($q) => $q->active()])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories,
        ]);
    });

    /* ============================== Downloads ============================= */
    Route::prefix('download')->group(function () {
        Route::get('{app:slug}/info', [DownloadController::class, 'apiInfo']);
        Route::post('{app:slug}/token', [DownloadController::class, 'generateToken']);
        Route::post('{app:slug}/verify', [DownloadController::class, 'verifyToken']);
    });

    /* ============================== Stats ================================= */
    Route::get('stats', function () {
        $stats = [
            'total_apps' => \App\Models\App::active()->count(),
            'total_categories' => \App\Models\Category::active()->count(),
            'total_downloads' => \App\Models\App::sum('downloads_count'),
            'latest_updated' => \App\Models\App::active()
                ->orderBy('updated_at', 'desc')
                ->first(['updated_at']),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
        ]);
    });
});

/* -------------------------------------------------------------------------- */
/* AUTHENTICATED API ROUTES (SANCTUM)                                         */
/* -------------------------------------------------------------------------- */
Route::prefix('v1')->middleware(['throttle:60,1', 'auth:sanctum'])->group(function () {
    Route::get('user', fn (Request $request) => response()->json([
        'success' => true,
        'data' => $request->user(),
    ]));
});

/* -------------------------------------------------------------------------- */
/* API DOCUMENTATION                                                          */
/* -------------------------------------------------------------------------- */
Route::get('docs', function () {
    return response()->json([
        'title' => 'APK Center API Documentation',
        'version' => '1.0.0',
        'base_url' => url('/api/v1'),
        'endpoints' => [
            'GET /apps/latest' => 'Get latest apps',
            'GET /apps/category/{slug}' => 'Get apps by category',
            'GET /apps/{package_name}' => 'Get single app details',
            'GET /apps/search?q={query}' => 'Search apps',
            'GET /categories' => 'List categories',
            'GET /download/{slug}/info' => 'Download info',
            'POST /download/{slug}/token' => 'Generate token',
            'POST /download/{slug}/verify' => 'Verify token',
            'GET /stats' => 'Platform statistics',
        ],
        'rate_limit' => '60 requests/minute',
        'authentication' => 'Bearer token for protected endpoints',
    ]);
});
Route::get('category/{category:slug}', [AppController::class, 'category'])->name('apps.category');
Route::get('app/{app:slug}', [AppController::class, 'show'])->name('apps.show');

/* =========================================================
| 3. DOWNLOAD ROUTES
|========================================================= */
Route::get('download/{app:slug}', [DownloadController::class, 'show'])->name('download.show');
Route::get('download/{app:slug}/process', [DownloadController::class, 'process'])->name('download.process');

/* =========================================================
| 4. STATIC PAGES
|========================================================= */
Route::view('contact', 'contact')->name('contact');
Route::view('about', 'pages.about')->name('about');
Route::view('terms', 'pages.terms')->name('terms');
Route::view('privacy', 'pages.privacy')->name('privacy');

/* =========================================================
| 5. ADMIN PANEL
|========================================================= */
Route::prefix('admin')->name('admin.')->group(function () {
    Route::get('/', [AdminController::class, 'index'])->name('dashboard');
    Route::post('import', [AdminController::class, 'runImport'])->name('import');
});

/* =========================================================
| 6. SYSTEM ROUTES
|========================================================= */
Route::get('sitemap.xml', [SitemapController::class, 'index'])->name('sitemap');

Route::get('robots.txt', function () {
    $content = "User-agent: *
";
    $content .= "Allow: /
";
    $content .= "Disallow: /api/
";
    $content .= "Disallow: /login
";
    $content .= "Disallow: /register
";
    $content .= "Sitemap: " . route('sitemap') . "
";

    return response($content)->header('Content-Type', 'text/plain');
});

/* LANGUAGE SWITCH */
Route::post('language', [LocalizationController::class, 'switch'])->name('language.switch');
Route::get('language/{locale}', [LocalizationController::class, 'switch'])
    ->name('language.switch.short')
    ->where('locale', '[a-z]{2}');

/* DARK MODE TOGGLE */
Route::get('dark-mode/toggle', function () {
    $isDark = session('dark_mode', false);
    session(['dark_mode' => ! $isDark]);
    return back();
})->name('dark-mode.toggle');

/* FALLBACK */
Route::fallback(function () {
    return response()->view('errors.404', [], 404);
});

