<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AppController;
use App\Http\Controllers\DownloadController;
use Illuminate\Http\Request;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// API Version 1
Route::prefix('v1')->middleware(['throttle:60,1'])->group(function () {

    // =========================================================
    // Apps endpoints
    // =========================================================
    Route::prefix('apps')->group(function () {

        // Get latest apps
        Route::get('latest', [AppController::class, 'apiLatest']);

        // Get apps by category (binding by slug is assumed if Category model has getRouteKeyName() = 'slug')
        Route::get('category/{category:slug}', [AppController::class, 'apiByCategory']);

        // Get single app by package name
        Route::get('{package_name}', [AppController::class, 'apiShow']);
        
        // Search apps
        Route::get('search', function (Request $request) {
            $query = $request->get('q');

            if (empty($query)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Search query is required',
                ], 400);
            }

            $apps = \App\Models\App::active()
                ->where(function ($q) use ($query) {
                    $q->where('name', 'LIKE', "%{$query}%")
                      ->orWhere('description', 'LIKE', "%{$query}%")
                      ->orWhere('package_name', 'LIKE', "%{$query}%");
                })
                ->with('category')
                ->paginate($request->get('per_page', 20));

            return response()->json([
                'success' => true,
                'data' => $apps->items(),
                'pagination' => [
                    'current_page' => $apps->currentPage(),
                    'last_page' => $apps->lastPage(),
                    'per_page' => $apps->perPage(),
                    'total' => $apps->total(),
                    'next_page_url' => $apps->nextPageUrl(),
                    'prev_page_url' => $apps->previousPageUrl(),
                ],
            ]);
        });

        // Get single app by package name
        Route::get('{package_name}', [AppController::class, 'apiShow'])
            ->where('package_name', '^(?!search$)(?!latest$)[\\w.-]+$');
    });

    // =========================================================
    // Categories endpoint
    // =========================================================
    Route::get('categories', function () {
        $categories = \App\Models\Category::active()
            ->ordered()
            ->withCount(['apps as apps_count' => fn ($q) => $q->active()])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories,
        ]);
    });

    // =========================================================
    // Download endpoints - تم تحديث الربط لـ {app:slug}
    // =========================================================
    Route::prefix('download')->group(function () {

        // Get download info (binding by slug)
        Route::get('{app:slug}/info', [DownloadController::class, 'apiInfo']);

        // Generate download token (binding by slug)
        Route::post('{app:slug}/token', [DownloadController::class, 'generateToken']);

        // Verify download token (binding by slug)
        Route::post('{app:slug}/verify', [DownloadController::class, 'verifyToken']);
    });

    // =========================================================
    // Stats endpoint
    // =========================================================
    Route::get('stats', function () {
        $stats = [
            'total_apps' => \App\Models\App::active()->count(),
            'total_categories' => \App\Models\Category::active()->count(),
            'total_downloads' => \App\Models\App::sum('downloads_count'),
            'latest_updated' => \App\Models\App::active()
                ->orderBy('updated_at', 'desc')
                ->first(['updated_at']),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
        ]);
    });
});

// =========================================================
// Authenticated API endpoints (require API key)
// =========================================================
Route::prefix('v1')->middleware(['throttle:60,1', 'auth:sanctum'])->group(function () {

    // Protected endpoints here
    Route::get('user', function (Request $request) {
        return response()->json([
            'success' => true,
            'data' => $request->user(),
        ]);
    });
});

// =========================================================
// API documentation
// =========================================================
Route::get('docs', function () {
    return response()->json([
        'title' => 'APK Center API Documentation',
        'version' => '1.0.0',
        'base_url' => url('/api/v1'),
        'endpoints' => [
            'GET /apps/latest' => 'Get latest 10 apps',
            'GET /apps/category/{slug}' => 'Get apps by category (supports pagination)',
            'GET /apps/{package_name}' => 'Get single app details',
            'GET /apps/search?q={query}' => 'Search apps',
            'GET /categories' => 'Get all categories',
            'GET /download/{slug}/info' => 'Get download information',
            'POST /download/{slug}/token' => 'Generate download token',
            'POST /download/{slug}/verify' => 'Verify download token',
            'GET /stats' => 'Get platform statistics',
        ],
        'rate_limit' => '60 requests per minute per IP',
        'authentication' => 'Bearer token required for protected endpoints',
    ]);
});