<?php

namespace App\Services;

use App\Models\App;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use ZipArchive;
use Exception;
use RuntimeException;

class ZipArchiveService
{
    private string $disk = 'public';
    private string $directory = 'exports';
    private string $apkDisk = 'local';
    private array $errors = [];

    public function __construct()
    {
        Storage::disk($this->disk)->makeDirectory($this->directory);
    }

    public function getErrors(): array
    {
        return $this->errors;
    }

    public function clearErrors(): void
    {
        $this->errors = [];
    }

    public function listArchives(): array
    {
        $files = Storage::disk($this->disk)->files($this->directory);
        $archives = [];

        foreach ($files as $file) {
            if (strtolower(pathinfo($file, PATHINFO_EXTENSION)) !== 'zip') {
                continue;
            }

            $archives[] = [
                'name' => basename($file),
                'size' => Storage::disk($this->disk)->size($file),
                'created' => Storage::disk($this->disk)->lastModified($file),
                'download_url' => Storage::disk($this->disk)->url($file),
            ];
        }

        usort($archives, fn (array $a, array $b) => $b['created'] <=> $a['created']);

        return $archives;
    }

    public function createFromAppCollection(string $collectionName, array $selectedApps, bool $includeApkFiles = true): string
    {
        $this->clearErrors();
        
        $safeName = Str::slug($collectionName) ?: 'collection';
        $fileName = sprintf('%s-%s.zip', $safeName, now()->format('YmdHis'));
        $relativePath = $this->directory . '/' . $fileName;
        $absolutePath = Storage::disk($this->disk)->path($relativePath);

        $zip = new ZipArchive();
        if ($zip->open($absolutePath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new RuntimeException('Unable to create ZIP archive.');
        }

        try {
            $apps = App::whereIn('id', $selectedApps)->get([
                'id', 'name', 'package_name', 'apk_path', 'download_url', 'latest_version'
            ]);
            
            if ($apps->isEmpty()) {
                throw new RuntimeException('No apps found for the provided selection.');
            }

            $zip->addFromString(
                'README.txt',
                $this->generateReadmeContent($collectionName, $apps)
            );

            foreach ($apps as $app) {
                $this->processApp($zip, $app, $includeApkFiles);
            }

            if (!empty($this->errors)) {
                $zip->addFromString(
                    'ERRORS.txt',
                    "The following errors occurred:\n\n" . implode("\n", $this->errors)
                );
            }

            $zip->close();
            return $absolutePath;

        } catch (Exception $e) {
            $zip->close();
            $this->cleanupFailedArchive($relativePath);
            throw $e;
        }
    }

    private function processApp(ZipArchive $zip, App $app, bool $includeApkFiles): void
    {
        $packageName = $app->package_name ?? 'unknown-' . $app->id;
        $safePackageName = Str::slug($packageName);

        $zip->addFromString(
            "{$safePackageName}/info.txt",
            $this->generateAppInfoContent($app)
        );

        if (!$includeApkFiles) {
            return;
        }

        $apkContent = $this->fetchApkContent($app);
        
        if ($apkContent !== null) {
            $apkFileName = "{$safePackageName}/{$safePackageName}.apk";
            $zip->addFromString($apkFileName, $apkContent);
            Log::info("Added APK for {$packageName} to archive");
        }
    }

    private function fetchApkContent(App $app): ?string
    {
        $packageName = $app->package_name ?? 'unknown';

        if (!empty($app->apk_path)) {
            $content = $this->fetchFromLocalStorage($app->apk_path, $packageName);
            if ($content !== null) {
                return $content;
            }
        }

        if (!empty($app->download_url)) {
            $content = $this->fetchFromUrl($app->download_url, $packageName);
            if ($content !== null) {
                return $content;
            }
        }

        $this->errors[] = "[{$packageName}] No APK file available";
        return null;
    }

    private function fetchFromLocalStorage(string $apkPath, string $packageName): ?string
    {
        $disks = [$this->apkDisk, 'public', 'local'];
        
        foreach ($disks as $disk) {
            try {
                if (Storage::disk($disk)->exists($apkPath)) {
                    $content = Storage::disk($disk)->get($apkPath);
                    if ($content !== false && !empty($content)) {
                        Log::info("Fetched APK for {$packageName} from {$disk}:{$apkPath}");
                        return $content;
                    }
                }
            } catch (Exception $e) {
                Log::warning("Failed to read APK from {$disk}:{$apkPath}: " . $e->getMessage());
            }
        }

        if (file_exists($apkPath) && is_readable($apkPath)) {
            $content = file_get_contents($apkPath);
            if ($content !== false) {
                return $content;
            }
        }

        $this->errors[] = "[{$packageName}] Local APK file not found: {$apkPath}";
        return null;
    }

    private function fetchFromUrl(string $url, string $packageName): ?string
    {
        try {
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                $this->errors[] = "[{$packageName}] Invalid download URL: {$url}";
                return null;
            }

            $response = Http::timeout(120)->withOptions(['verify' => false])->get($url);

            if ($response->successful()) {
                $content = $response->body();
                
                if (strlen($content) < 1000) {
                    $this->errors[] = "[{$packageName}] Downloaded file too small: {$url}";
                    return null;
                }

                Log::info("Downloaded APK for {$packageName} from URL: {$url}");
                return $content;
            }

            $this->errors[] = "[{$packageName}] HTTP {$response->status()} from: {$url}";
            return null;

        } catch (Exception $e) {
            $this->errors[] = "[{$packageName}] Download failed: " . $e->getMessage();
            return null;
        }
    }

    private function generateReadmeContent(string $collectionName, Collection $apps): string
    {
        $content = "Exported Apps Collection: {$collectionName}\n";
        $content .= "Generated at: " . now()->toDateTimeString() . "\n";
        $content .= "Total apps: " . $apps->count() . "\n\nIncluded Apps:\n";
        
        foreach ($apps as $index => $app) {
            $num = $index + 1;
            $content .= "{$num}. {$app->name} ({$app->package_name})\n";
        }
        
        return $content;
    }

    private function generateAppInfoContent(App $app): string
    {
        return "Name: {$app->name}\nPackage: {$app->package_name}\nVersion: " . ($app->latest_version ?? 'N/A') . "\nAPK Path: " . ($app->apk_path ?? 'N/A') . "\nDownload URL: " . ($app->download_url ?? 'N/A') . "\n";
    }

    private function cleanupFailedArchive(string $relativePath): void
    {
        try {
            if (Storage::disk($this->disk)->exists($relativePath)) {
                Storage::disk($this->disk)->delete($relativePath);
            }
        } catch (Exception $e) {
            Log::error("Failed to cleanup archive: " . $e->getMessage());
        }
    }

    public function getArchivePath(string $fileName): ?string
    {
        $relativePath = $this->directory . '/' . basename($fileName);

        if (!Storage::disk($this->disk)->exists($relativePath)) {
            return null;
        }

        return Storage::disk($this->disk)->path($relativePath);
    }

    public function cleanupOldArchives(int $days): int
    {
        $threshold = now()->subDays($days)->getTimestamp();
        $deleted = 0;

        foreach (Storage::disk($this->disk)->files($this->directory) as $file) {
            if (strtolower(pathinfo($file, PATHINFO_EXTENSION)) !== 'zip') {
                continue;
            }

            if (Storage::disk($this->disk)->lastModified($file) < $threshold) {
                Storage::disk($this->disk)->delete($file);
                $deleted++;
            }
        }

        return $deleted;
    }
}
