<?php

namespace App\Services;

use App\Models\App;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

/**
 * SEO optimization service.
 * Generates sitemaps, meta tags, structured data, and OpenGraph tags.
 */
class SEOOptimizer
{
    /**
     * Generate XML sitemap for all apps and pages.
     *
     * @return string XML sitemap content
     */
    public static function generateSitemap(): string
    {
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . PHP_EOL;

        // Add static pages
        $staticPages = [
            '',
            '/about',
            '/contact',
            '/privacy',
            '/terms',
        ];

        foreach ($staticPages as $page) {
            $url = url($page);
            $xml .= self::sitemapEntry($url, 'weekly', '1.0');
        }

        // Add all app pages
        App::where('status', 'active')
            ->select('slug', 'updated_at')
            ->chunk(500, function ($apps) use (&$xml) {
                foreach ($apps as $app) {
                    $url = url('/app/' . $app->slug);
                    $lastmod = $app->updated_at->format('Y-m-d');
                    $xml .= '  <url>' . PHP_EOL;
                    $xml .= '    <loc>' . htmlspecialchars($url) . '</loc>' . PHP_EOL;
                    $xml .= '    <lastmod>' . $lastmod . '</lastmod>' . PHP_EOL;
                    $xml .= '    <changefreq>weekly</changefreq>' . PHP_EOL;
                    $xml .= '    <priority>0.8</priority>' . PHP_EOL;
                    $xml .= '  </url>' . PHP_EOL;
                }
            });

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Helper to build sitemap entry.
     *
     * @param string $loc
     * @param string $changefreq
     * @param string $priority
     * @return string
     */
    private static function sitemapEntry(string $loc, string $changefreq, string $priority): string
    {
        return '  <url>' . PHP_EOL
            . '    <loc>' . htmlspecialchars($loc) . '</loc>' . PHP_EOL
            . '    <changefreq>' . $changefreq . '</changefreq>' . PHP_EOL
            . '    <priority>' . $priority . '</priority>' . PHP_EOL
            . '  </url>' . PHP_EOL;
    }

    /**
     * Save sitemap to public/sitemap.xml
     */
    public static function saveSitemap(): void
    {
        $sitemap = self::generateSitemap();
        Storage::disk('public')->put('sitemap.xml', $sitemap);
    }

    /**
     * Generate JSON-LD structured data for an app (for Google Rich Results).
     *
     * @param App $app
     * @return array
     */
    public static function generateAppSchema(App $app): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'SoftwareApplication',
            'name' => $app->name,
            'description' => $app->description ?: 'Download ' . $app->name . ' APK',
            'url' => url('/app/' . $app->slug),
            'image' => $app->icon_url ?: url('/images/default-app-icon.png'),
            'applicationCategory' => 'Utility',
            'operatingSystem' => 'Android',
            'softwareVersion' => $app->latest_version,
            'downloadUrl' => $app->download_url,
            'fileSize' => $app->size ? self::formatBytes($app->size) : null,
            'author' => [
                '@type' => 'Organization',
                'name' => $app->developer ?: 'Unknown Developer',
            ],
            'aggregateRating' => $app->rating > 0 ? [
                '@type' => 'AggregateRating',
                'ratingValue' => number_format($app->rating, 1),
                'ratingCount' => $app->downloads_count,
            ] : null,
            'offers' => [
                '@type' => 'Offer',
                'price' => '0',
                'priceCurrency' => 'USD',
            ],
        ];
    }

    /**
     * Generate OpenGraph meta tags for social sharing.
     *
     * @param App $app
     * @return array
     */
    public static function generateOpenGraphTags(App $app): array
    {
        return [
            'og:title' => $app->name,
            'og:description' => $app->description ?: 'Download ' . $app->name . ' for Android',
            'og:image' => $app->icon_url ?: url('/images/default-app-icon.png'),
            'og:url' => url('/app/' . $app->slug),
            'og:type' => 'website',
            'twitter:card' => 'summary_large_image',
            'twitter:title' => $app->name,
            'twitter:description' => $app->description ?: 'Download ' . $app->name,
            'twitter:image' => $app->icon_url ?: url('/images/default-app-icon.png'),
        ];
    }

    /**
     * Generate SEO meta tags for an app.
     *
     * @param App $app
     * @return array
     */
    public static function generateMetaTags(App $app): array
    {
        $keywords = implode(', ', [
            $app->name . ' APK',
            'Download ' . $app->name,
            $app->category->name ?? 'Apps',
            'Android Apps',
        ]);

        return [
            'title' => $app->name . ' APK Download - ' . config('app.name'),
            'description' => substr($app->description ?: 'Download ' . $app->name . ' APK for Android', 0, 160),
            'keywords' => $keywords,
            'canonical' => url('/app/' . $app->slug),
        ];
    }

    /**
     * Format bytes to human-readable format.
     *
     * @param int $bytes
     * @return string
     */
    private static function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= (1 << (10 * $pow));

        return round($bytes, 2) . ' ' . $units[$pow];
    }

    /**
     * Get all categories with counts for breadcrumb/schema data.
     *
     * @return array
     */
    public static function getCategoryHierarchy(): array
    {
        return \App\Models\Category::with('apps')
            ->get()
            ->map(fn ($cat) => [
                'name' => $cat->name,
                'url' => url('/category/' . $cat->slug),
                'count' => $cat->apps_count ?? 0,
            ])
            ->toArray();
    }
}
