<?php

namespace App\Services;

use App\Contracts\ImportSourceParser;
use RuntimeException;

class ImportSourceManager
{
    /**
     * Cache of configured sources.
     *
     * @var array<string, array>
     */
    protected array $configuredSources;

    public function __construct()
    {
        $this->configuredSources = config('import_sources.sources', []);
    }

    /**
     * Get all configured import sources.
     *
     * @return array<string, array>
     */
    public function all(): array
    {
        return $this->configuredSources;
    }

    /**
     * Determine if the given source exists.
     */
    public function has(string $key): bool
    {
        return isset($this->configuredSources[$key]);
    }

    /**
     * Resolve the parser instance for a specific source.
     */
    public function parser(string $key): ImportSourceParser
    {
        $source = $this->configuredSources[$key] ?? null;

        if ($source === null || empty($source['parser'])) {
            throw new RuntimeException("Import parser not configured for source '{$key}'.");
        }

        return app()->make($source['parser'], [
            'config' => $source,
        ]);
    }

    /**
     * Get the raw configuration for a source.
     */
    public function config(string $key): array
    {
        return $this->configuredSources[$key] ?? [];
    }
}
