<?php

namespace App\Orchid\Screens;

use App\Models\App;
use App\Services\ZipArchiveService;
use Illuminate\Http\Request;
use Orchid\Screen\Screen;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Alert;
use Orchid\Screen\Fields\Input;
use Orchid\Screen\Fields\Relation;
use Orchid\Screen\Actions\Button;
use Orchid\Screen\TD;

class ExportZipScreen extends Screen
{
    public $name = 'Export Collections as ZIP';
    public $description = 'Create ZIP archives of app collections for backup or sharing';

    public function query(): iterable
    {
        $zipService = new ZipArchiveService();
        $archives = $zipService->listArchives();

        return [
            'apps' => App::where('status', 'active')
                ->select('id', 'name', 'package_name')
                ->orderBy('name')
                ->get(),
            'archives' => $archives,
        ];
    }

    public function commandBar(): iterable
    {
        return [
            Button::make('Back to Dashboard')
                ->icon('bs.arrow-left')
                ->link()
                ->route('platform.dashboard'),

            Button::make('Clean Old Archives')
                ->icon('bs.trash')
                ->method('cleanupArchives')
                ->confirm('Delete archives older than 7 days?')
                ->class('btn-warning'),
        ];
    }

    public function layout(): iterable
    {
        return [
            // Create New Collection Archive
            Layout::block([
                Layout::rows([
                    Input::make('collection_name')
                        ->title('Collection Name')
                        ->placeholder('e.g., My Favorite Apps, Games Collection')
                        ->required(),

                    Relation::make('selected_apps')
                        ->fromModel(App::class, 'name', 'id')
                        ->multiple()
                        ->title('Select Apps to Include')
                        ->required()
                        ->help('Select apps to include in the ZIP'),

                    Button::make('Create ZIP Archive')
                        ->icon('bs.file-earmark-zip')
                        ->method('createArchive')
                        ->class('btn-primary'),
                ]),
            ])->title('Create New Collection Archive'),

            // Existing Archives Table
            Layout::block([
                Layout::table('archives', [
                    TD::make('name', 'Filename'),
                    
                    TD::make('size', 'Size')
                        ->render(function (array $item) {
                            return $this->formatBytes($item['size'] ?? 0);
                        }),
                    
                    TD::make('created', 'Created')
                        ->render(function (array $item) {
                            return date('Y-m-d H:i:s', $item['created']);
                        }),
                    
                    TD::make('download', 'Action')
                        ->render(function (array $item) {
                            return '
                                <a href="' . $item['download_url'] . '" class="btn btn-sm btn-info">
                                    <i class="icon-download"></i> Download
                                </a>
                                <form method="POST" action="' . route('platform.export-zip.delete') . '" style="display:inline;">
                                    ' . csrf_field() . '
                                    <input type="hidden" name="file_name" value="' . htmlspecialchars($item['name']) . '">
                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm(\'Delete this archive?\')">
                                        <i class="icon-trash"></i> Delete
                                    </button>
                                </form>
                            ';
                        }),
                ]),
            ])->title('Existing Archives'),
        ];
    }

    /**
     * Create a ZIP archive from selected apps.
     */
    public function createArchive()
    {
        $collectionName = request('collection_name');
        $selectedApps = request('selected_apps', []);

        if (empty($collectionName) || empty($selectedApps)) {
            Alert::warning('Please enter a collection name and select at least one app');
            return redirect()->route('platform.export-zip');
        }

        try {
            $zipService = new ZipArchiveService();
            $zipPath = $zipService->createFromAppCollection($collectionName, $selectedApps);

            $fileName = basename($zipPath);
            Alert::success('Archive created successfully: ' . $fileName);

            return redirect()->route('platform.export-zip');
        } catch (\Exception $e) {
            Alert::danger('Failed to create archive: ' . $e->getMessage());
            return redirect()->route('platform.export-zip');
        }
    }

    /**
     * Delete a ZIP archive.
     */
    public function deleteArchive()
    {
        $fileName = request('file_name');

        if (empty($fileName)) {
            Alert::warning('No file specified');
            return redirect()->route('platform.export-zip');
        }

        try {
            $zipService = new ZipArchiveService();
            $filePath = $zipService->getArchivePath($fileName);

            if (!$filePath) {
                Alert::warning('Archive not found');
                return redirect()->route('platform.export-zip');
            }

            \Illuminate\Support\Facades\File::delete($filePath);
            Alert::success('Archive deleted: ' . $fileName);

            return redirect()->route('platform.export-zip');
        } catch (\Exception $e) {
            Alert::danger('Failed to delete archive: ' . $e->getMessage());
            return redirect()->route('platform.export-zip');
        }
    }

    /**
     * Clean up old archives.
     */
    public function cleanupArchives()
    {
        try {
            $zipService = new ZipArchiveService();
            $deleted = $zipService->cleanupOldArchives(7);

            Alert::success('Cleaned up ' . $deleted . ' old archive(s)');
            return redirect()->route('platform.export-zip');
        } catch (\Exception $e) {
            Alert::danger('Cleanup failed: ' . $e->getMessage());
            return redirect()->route('platform.export-zip');
        }
    }

    /**
     * Format bytes to human-readable format.
     *
     * @param int $bytes
     * @return string
     */
    private function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= (1 << (10 * $pow));

        return round($bytes, 2) . ' ' . $units[$pow];
    }
}
