<?php

namespace App\Orchid\Screens;

use Illuminate\Http\Request;
use App\Models\App;
use App\Models\Category;
use App\Services\MultiSourceScraper;
use Illuminate\Support\Str;
use Orchid\Screen\Screen;
use Orchid\Support\Facades\Layout;
use Orchid\Support\Facades\Alert;
use Orchid\Screen\Fields\Input;
use Orchid\Screen\Fields\Select;
use Orchid\Screen\Fields\TextArea;
use Orchid\Screen\Fields\CheckBox;
use Orchid\Screen\Actions\Button;

class APKImportScreen extends Screen
{
    public $name = 'Import APKs';
    public $description = 'Scrape and import apps from multiple sources';

    public function query(): iterable
    {
        return [
            'drafts'     => session('drafts', []),
            'sources'    => [
                'github'    => 'GitHub Releases',
                'apkpure'   => 'APKPure (Web Scrape)',
                'apkmirror' => 'APKMirror (Web Scrape)',
            ],
            'categories' => Category::pluck('name', 'id')->toArray(),
        ];
    }

    public function commandBar(): iterable
    {
        return [
            Button::make('Back to Dashboard')
                ->icon('bs.arrow-left')
                ->route('platform.dashboard'),
        ];
    }

    public function layout(): iterable
    {
        return [
            // 1. البحث
            Layout::block(
                Layout::rows([
                    Select::make('source')
                        ->title('Source')
                        ->options([
                            'github'    => 'GitHub Releases',
                            'apkpure'   => 'APKPure (Web Scrape)',
                            'apkmirror' => 'APKMirror (Web Scrape)',
                        ])
                        ->required()
                        ->help('Choose a source to scrape from. GitHub Releases is the most reliable.'),

                    Input::make('query')
                        ->title('Search Query')
                        ->placeholder('For GitHub: owner/repo (e.g., mozilla-mobile/fenix)')
                        ->required()
                        ->help('App name, package name, or GitHub repo path'),

                    Button::make('Search & Scrape')
                        ->icon('bs.search')
                        ->method('scrapeApps')
                        ->class('btn btn-primary'),
                ])
            )->title('1. Find APKs'),

            // 2. عرض النتائج عبر Blade view
            Layout::block(
                Layout::view('orchid.apk-import-table')
            )->title('2. Review Scraped Results'),

            // 3. الأكشنات الجماعية
            Layout::block(
                Layout::rows([
                    Select::make('category_id')
    ->title('Default Category')
    ->options(Category::pluck('name', 'id')->toArray())
    ->empty('Select category', '')
    ->help('Category to assign to imported apps if not specified'),

                    CheckBox::make('set_draft_status')
                        ->title('Import as Draft')
                        ->value(1)
                        ->help('Check to import apps with "draft" status (unpublished)'),

                    TextArea::make('notes')
                        ->title('Import Notes')
                        ->placeholder('Any notes about this import batch...')
                        ->rows(3),

                    Button::make('Bulk Publish All')
                        ->icon('bs.check-all')
                        ->method('bulkPublish')
                        ->class('btn btn-success')
                        ->confirm('Publish all scraped apps?'),
                ])
            )->title('3. Bulk Actions'),
        ];
    }

    /**
     * Scrape apps from the selected source.
     */
    public function scrapeApps()
    {
        $source = request('source');
        $query  = request('query');

        if (empty($source) || empty($query)) {
            Alert::warning('Choose a source and enter a search query before scraping.');

            return redirect()->route('platform.apk-import');
        }

        // فقط لتسهيل التجربة مع GitHub
        if ($source === 'github' && empty($query)) {
            $query = 'mozilla-mobile/fenix';
        }

        try {
            $drafts = MultiSourceScraper::scrape($source, $query);

            $validDrafts = [];
            foreach ($drafts as $draft) {
                $errors = MultiSourceScraper::validate($draft);
                if (empty($errors)) {
                    $validDrafts[] = $draft;
                }
            }

            session(['drafts' => $validDrafts]);

            if (empty($validDrafts)) {
                Alert::warning('Scrape completed but returned no valid apps. Try a different query or source.');
            } else {
                Alert::success('Found '.count($validDrafts).' valid apps to import');
            }

            return redirect()
                ->route('platform.apk-import')
                ->with([
                    'source' => $source,
                    'query'  => $query,
                ]);
        } catch (\Throwable $e) {
            logger()->error('APK import scrape failed', [
                'source'  => $source,
                'query'   => $query,
                'message' => $e->getMessage(),
            ]);
            Alert::error('Scrape failed: '.$e->getMessage());

            return redirect()->route('platform.apk-import');
        }
    }

    /**
     * Publish a single scraped app.
     */
    public function publishApp()
    {
        $appData = request('app_data');

        if (is_string($appData)) {
            $appData = json_decode($appData, true);
        }

        $errors = MultiSourceScraper::validate($appData);
        if (! empty($errors)) {
            Alert::error('Validation failed: '.implode(', ', $errors));

            return redirect()->route('platform.apk-import');
        }

        $existing = App::where('package_name', $appData['package_name'])->first();
        if ($existing) {
            Alert::warning('App "'.$appData['package_name'].'" already exists. Skipped.');

            return redirect()->route('platform.apk-import');
        }

        try {
            $appData['status']      = request('set_draft_status') ? 'draft' : 'active';
            $appData['category_id'] = request('category_id', 1);
            $appData['slug']        = Str::slug($appData['name']);

            App::create($appData);

            Alert::success('App published: '.$appData['name']);

            return redirect()->route('platform.apk-import');
        } catch (\Exception $e) {
            Alert::error('Publish failed: '.$e->getMessage());

            return redirect()->route('platform.apk-import');
        }
    }

    /**
     * Bulk publish all validated drafts.
     */
    public function bulkPublish()
    {
        $drafts = session('drafts', []);

        if (empty($drafts)) {
            Alert::warning('No drafts to publish');

            return redirect()->route('platform.apk-import');
        }

        $categoryId = request('category_id', 1);
        $published  = 0;
        $skipped    = 0;

        foreach ($drafts as $draft) {
            if (App::where('package_name', $draft['package_name'])->exists()) {
                $skipped++;
                continue;
            }

            $draft['status']      = request('set_draft_status') ? 'draft' : 'active';
            $draft['category_id'] = $categoryId;
            $draft['slug']        = Str::slug($draft['name']);

            try {
                App::create($draft);
                $published++;
            } catch (\Exception $e) {
                $skipped++;
            }
        }

        Alert::success("Bulk publish complete: $published published, $skipped skipped");

        return redirect()->route('platform.apk-import');
    }
}