<?php

namespace App\Imports\Scrapers;

use App\Contracts\ImportSourceParser;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class JsonFeedScraper implements ImportSourceParser
{
    /**
     * Cache of the decoded payload.
     *
     * @var array
     */
    protected array $payload = [];

    /**
     * Create a new parser instance.
     */
    public function __construct(
        protected array $config = []
    ) {}

    /**
     * {@inheritDoc}
     */
    public function getCategories(): array
    {
        $payload = $this->payload();

        $results = [];

        foreach ($payload['categories'] ?? [] as $category) {
            $slug = $category['slug'] ?? Str::slug($category['name'] ?? 'category');

            $results[$slug] = [
                'name' => $category['name'] ?? ucfirst($slug),
                'slug' => $slug,
                'description' => $category['description'] ?? null,
                'icon' => $category['icon'] ?? null,
                'color' => $category['color'] ?? ($this->config['brand_color'] ?? null),
            ];
        }

        if (empty($results)) {
            $defaultCategory = $this->config['default_category'] ?? 'community';
            $results[$defaultCategory] = [
                'name' => ucfirst($defaultCategory),
                'slug' => Str::slug($defaultCategory),
                'description' => $this->config['description'] ?? null,
                'icon' => $this->config['icon'] ?? null,
                'color' => $this->config['brand_color'] ?? null,
            ];
        }

        return array_values($results);
    }

    /**
     * {@inheritDoc}
     */
    public function getApps(): array
    {
        $payload = $this->payload();

        $defaultCategory = $this->config['default_category'] ?? 'community';

        return collect($payload['apps'] ?? [])->map(function ($app) use ($defaultCategory) {
            $categorySlug = Str::slug($app['category'] ?? $defaultCategory);

            $app['category'] = [
                'name' => $app['category_name'] ?? ucfirst($categorySlug),
                'slug' => $categorySlug,
            ];

            return $app;
        })->all();
    }

    /**
     * Load the feed payload once.
     */
    protected function payload(): array
    {
        if ($this->payload !== []) {
            return $this->payload;
        }

        $feedUrl = $this->config['feed_url'] ?? '';

        if (empty($feedUrl)) {
            return $this->payload = [];
        }

        if (str_starts_with($feedUrl, 'http')) {
            $response = Http::get($feedUrl);

            if (! $response->successful()) {
                return $this->payload = [];
            }

            return $this->payload = $response->json() ?? [];
        }

        if (File::exists($feedUrl)) {
            $content = File::get($feedUrl);
            return $this->payload = json_decode($content, true) ?? [];
        }

        return $this->payload = [];
    }
}
