<?php

namespace App\Http\Controllers;

use App\Models\App;
use App\Models\Category;
use Illuminate\Support\Facades\Response;
use Illuminate\Database\Eloquent\Collection; // Use for type hinting clarity

class SitemapController extends Controller
{
    /**
     * Generate dynamic sitemap.xml to be consumed by search engines.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        // 1. استخدام التخزين المؤقت (Buffering) لتحسين أداء السلاسل النصية الكبيرة
        ob_start();

        echo '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . PHP_EOL;

        // Add homepage
        echo $this->addUrl(route('home'), now()->toAtomString(), 'weekly', '1.0');

        // Add categories (using cursor() for memory efficiency)
        $categories = Category::select('slug', 'updated_at')->cursor();
        foreach ($categories as $category) {
            echo $this->addUrl(
                route('apps.category', $category->slug),
                $category->updated_at?->toAtomString() ?? now()->toAtomString(),
                'weekly',
                '0.8'
            );
        }

        // Add apps (using cursor() for memory efficiency on potentially large tables)
        $apps = App::select('slug', 'updated_at')->cursor();
        foreach ($apps as $app) {
            echo $this->addUrl(
                route('apps.show', $app->slug),
                $app->updated_at?->toAtomString() ?? now()->toAtomString(),
                'daily',
                '0.9'
            );
        }

        echo '</urlset>';

        $sitemap = ob_get_clean(); // Get the content and clean the buffer

        return Response::make($sitemap, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Helper function to format a single URL entry for the sitemap.
     *
     * @param string $loc The URL location.
     * @param string $lastmod The date of last modification (in Atom format).
     * @param string $changefreq The frequency of changes (always, hourly, daily, etc.).
     * @param string $priority The priority of this URL (0.0 to 1.0).
     * @return string The formatted XML <url> entry.
     */
    private function addUrl(string $loc, string $lastmod, string $changefreq, string $priority): string
    {
        return "  <url>\n"
            . "    <loc>" . htmlspecialchars($loc) . "</loc>\n"
            . "    <lastmod>" . $lastmod . "</lastmod>\n"
            . "    <changefreq>" . $changefreq . "</changefreq>\n"
            . "    <priority>" . $priority . "</priority>\n"
            . "  </url>\n";
    }
}
