<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\MultiSourceScraper;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Validator;
use Throwable;

class ScraperController extends Controller
{
    /**
     * Allow admins to scrape APK drafts directly from the dashboard.
     */
    public function search(Request $request)
    {
        $sources = MultiSourceScraper::getSourceInfo();
        $validated = Validator::make($request->all(), [
            'source' => ['nullable', 'string'],
            'query' => ['nullable', 'string', 'max:255'],
        ])->validate();

        $selectedSource = $validated['source'] ?? null;
        $query = $validated['query'] ?? null;

        $drafts = [];
        $statusMessage = null;
        $errorMessage = null;

        if ($selectedSource && $query) {
            if (! isset($sources[$selectedSource])) {
                return back()
                    ->withInput()
                    ->withErrors(['source' => __('The selected source is not supported.')]);
            }

            try {
                $rawDrafts = MultiSourceScraper::scrape($selectedSource, $query);

                $drafts = array_map(function (array $draft) {
                    $draft['validation_errors'] = MultiSourceScraper::validate($draft);

                    return $draft;
                }, $rawDrafts);

                $validDrafts = array_values(array_filter(
                    $drafts,
                    fn (array $draft) => empty($draft['validation_errors'])
                ));

                if (! empty($validDrafts)) {
                    session([
                        'drafts' => array_map(
                            fn (array $draft) => Arr::except($draft, ['validation_errors']),
                            $validDrafts
                        ),
                    ]);
                }

                $statusMessage = __('Found :count draft(s) from :source.', [
                    'count' => count($drafts),
                    'source' => $sources[$selectedSource]['name'] ?? $selectedSource,
                ]);
            } catch (Throwable $exception) {
                $errorMessage = $exception->getMessage();
            }
        }

        return view('admin.apk-import', [
            'sources' => $sources,
            'selectedSource' => $selectedSource,
            'query' => $query,
            'drafts' => $drafts,
            'statusMessage' => $statusMessage,
            'errorMessage' => $errorMessage,
            'sessionDrafts' => session('drafts', []),
            'editingDraft' => null,
            'editingIndex' => null,
        ]);
    }

    public function editDraft(int $index)
    {
        $sessionDrafts = session('drafts', []);

        if (! array_key_exists($index, $sessionDrafts)) {
            abort(404);
        }

        return view('admin.apk-import', [
            'sources' => MultiSourceScraper::getSourceInfo(),
            'selectedSource' => null,
            'query' => null,
            'drafts' => [],
            'statusMessage' => null,
            'errorMessage' => null,
            'sessionDrafts' => $sessionDrafts,
            'editingDraft' => $sessionDrafts[$index],
            'editingIndex' => $index,
        ]);
    }

    public function updateDraft(Request $request, int $index)
    {
        $sessionDrafts = session('drafts', []);

        if (! array_key_exists($index, $sessionDrafts)) {
            abort(404);
        }

        $validated = Validator::make($request->all(), [
            'name' => ['required', 'string', 'min:2', 'max:100'],
            'package_name' => ['required', 'regex:/^[a-z][a-z0-9_]*(\.[a-z0-9_]+)*$/i'],
            'developer' => ['required', 'string', 'max:255'],
            'latest_version' => ['required', 'string', 'max:50'],
            'download_url' => ['required', 'url'],
            'rating' => ['nullable', 'numeric', 'between:0,5'],
            'size' => ['nullable', 'integer', 'min:1024', 'max:2147483648'],
            'source_url' => ['nullable', 'url'],
            'source' => ['nullable', 'string', 'max:50'],
        ])->validate();

        $validationErrors = MultiSourceScraper::validate($validated);

        if (! empty($validationErrors)) {
            return back()->withInput()->withErrors($validationErrors);
        }

        $sessionDrafts[$index] = array_merge($sessionDrafts[$index], $validated);

        session(['drafts' => $sessionDrafts]);

        return redirect()->route('admin.apk-import')->with(
            'status',
            __('Draft updated successfully.')
        );
    }
}