<?php

namespace App\Helpers;

use App\Models\App;
use App\Services\SEOOptimizer;

/**
 * SEO helper for rendering meta tags and structured data in Blade templates.
 */
class SEOHelper
{
    /**
     * Render meta tags for an app page.
     *
     * @param App $app
     * @return string HTML meta tags
     */
    public static function renderAppMeta(App $app): string
    {
        $meta = SEOOptimizer::generateMetaTags($app);
        $og = SEOOptimizer::generateOpenGraphTags($app);
        $schema = SEOOptimizer::generateAppSchema($app);

        $html = [];

        // Standard meta tags
        $html[] = '<title>' . htmlspecialchars($meta['title']) . '</title>';
        $html[] = '<meta name="description" content="' . htmlspecialchars($meta['description']) . '">';
        $html[] = '<meta name="keywords" content="' . htmlspecialchars($meta['keywords']) . '">';
        $html[] = '<link rel="canonical" href="' . htmlspecialchars($meta['canonical']) . '">';

        // Open Graph tags
        foreach ($og as $key => $value) {
            if ($value) {
                $html[] = '<meta property="' . htmlspecialchars($key) . '" content="' . htmlspecialchars($value) . '">';
            }
        }

        // JSON-LD structured data
        $schemaFiltered = array_filter($schema);
        $html[] = '<script type="application/ld+json">' . json_encode($schemaFiltered, JSON_UNESCAPED_SLASHES) . '</script>';

        return implode(PHP_EOL, $html);
    }

    /**
     * Render breadcrumb schema for category pages.
     *
     * @param array $breadcrumbs Array of ['name' => '', 'url' => '']
     * @return string JSON-LD script
     */
    public static function renderBreadcrumbSchema(array $breadcrumbs): string
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => [],
        ];

        foreach ($breadcrumbs as $index => $crumb) {
            $schema['itemListElement'][] = [
                '@type' => 'ListItem',
                'position' => $index + 1,
                'name' => $crumb['name'],
                'item' => $crumb['url'],
            ];
        }

        return '<script type="application/ld+json">' . json_encode($schema, JSON_UNESCAPED_SLASHES) . '</script>';
    }

    /**
     * Render organization schema.
     *
     * @return string JSON-LD script
     */
    public static function renderOrganizationSchema(): string
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => config('app.name', 'APKs Center'),
            'url' => url('/'),
            'logo' => url('/images/logo.png'),
            'description' => 'Download free APK files for Android apps',
            'sameAs' => [
                'https://facebook.com/apkscenter',
                'https://twitter.com/apkscenter',
            ],
        ];

        return '<script type="application/ld+json">' . json_encode($schema, JSON_UNESCAPED_SLASHES) . '</script>';
    }

    /**
     * Render collection page schema.
     *
     * @param array $items Array of items with 'name', 'url', 'image'
     * @return string JSON-LD script
     */
    public static function renderCollectionSchema(array $items): string
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'ItemList',
            'itemListElement' => [],
        ];

        foreach ($items as $index => $item) {
            $schema['itemListElement'][] = [
                '@type' => 'ListItem',
                'position' => $index + 1,
                'name' => $item['name'],
                'url' => $item['url'],
                'image' => $item['image'] ?? null,
            ];
        }

        return '<script type="application/ld+json">' . json_encode($schema, JSON_UNESCAPED_SLASHES) . '</script>';
    }
}
