<?php

namespace App\Console;

use App\Console\Commands\ScrapeApps;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * The Artisan commands provided by the application.
     *
     * @var array
     */
    protected $commands = [
        ScrapeApps::class,
    ];

    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Run the import command every 6 hours
        $schedule->command('app:import-new')
                 ->everySixHours()
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->appendOutputTo(storage_path('logs/import.log'))
                 ->emailOutputOnFailure(config('app.admin_email', 'admin@example.com'));

        // Clean up old log files weekly
        $schedule->command('log:clear')
                 ->weekly()
                 ->at('02:00')
                 ->withoutOverlapping();

        // Cache cleanup daily
        $schedule->command('cache:clear')
                 ->daily()
                 ->at('03:00')
                 ->withoutOverlapping();

        // Database backup daily (if configured)
        $schedule->command('db:backup')
                 ->daily()
                 ->at('01:00')
                 ->withoutOverlapping()
                 ->when(function () {
                     return config('backup.enabled', false);
                 });

        // Generate sitemap daily
        $schedule->command('sitemap:generate')
                 ->daily()
                 ->at('04:00')
                 ->withoutOverlapping();

        // Health check every 5 minutes
        $schedule->command('health:check')
                 ->everyFiveMinutes()
                 ->withoutOverlapping()
                 ->runInBackground()
                 ->onFailure(function () {
                     // Send notification on health check failure
                     logger()->critical('Application health check failed');
                 });

        // Optimize database weekly
        $schedule->command('db:optimize')
                 ->weekly()
                 ->at('05:00')
                 ->withoutOverlapping();

        // Generate performance report monthly
        $schedule->command('report:performance')
                 ->monthly()
                 ->at('06:00')
                 ->withoutOverlapping()
                 ->emailOutputTo(config('app.admin_email', 'admin@example.com'));
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }

    /**
     * Get the timezone that should be used by default for scheduled events.
     */
    protected function scheduleTimezone(): string
    {
        return config('app.timezone', 'UTC');
    }
}