<?php

namespace App\Console\Commands;

use App\Services\MultiSourceScraper;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Throwable;

class ScrapeApps extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'scrape:apps
                            {source : Source to scrape (apkpure, apkmirror, github, fdroid)}
                            {query : Search query or identifier}
                            {--output= : Optional path (relative to storage/app) to save the results as JSON}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Scrape app metadata from supported sources and output or save the normalized results.';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $source = (string) $this->argument('source');
        $query  = (string) $this->argument('query');
        $output = $this->option('output');

        $this->info("Scraping {$source} for '{$query}'...");

        try {
            $results = MultiSourceScraper::scrape($source, $query);
        } catch (Throwable $e) {
            $this->error('Scraping failed: '.$e->getMessage());

            return Command::FAILURE;
        }

        $normalizedResults = array_map(fn (array $app) => $this->normalizeResult($app), $results);

        if ($output) {
            $this->saveResults($normalizedResults, $output);

            return Command::SUCCESS;
        }

        if (empty($normalizedResults)) {
            $this->warn('No results found.');

            return Command::SUCCESS;
        }

        $this->table(
            ['Package', 'Name', 'Version', 'Download URL'],
            array_map(fn ($app) => [
                $app['package'],
                $app['name'],
                $app['version'],
                $app['download_url'] ?? 'N/A',
            ], $normalizedResults)
        );

        return Command::SUCCESS;
    }

    /**
     * Normalize scraper output for console/JSON presentation.
     */
    private function normalizeResult(array $app): array
    {
        $version = $app['latest_version'] ?? $app['version'] ?? '';

        return [
            'package'      => $app['package_name'] ?? '',
            'name'         => $app['name'] ?? '',
            'version'      => (string) $version,
            'download_url' => $app['download_url'] ?? null,
        ];
    }

    /**
     * Persist results to local storage.
     */
    private function saveResults(array $results, string $path): void
    {
        $trimmedPath = ltrim($path, '/');

        Storage::disk('local')->put(
            $trimmedPath,
            json_encode($results, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );

        $this->info(sprintf('Saved %d record(s) to storage/app/%s', count($results), $trimmedPath));
    }
}