<?php

namespace App\Console\Commands;

use App\Models\App;
use App\Models\Category;
use App\Services\ImportSourceManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AppImportNew extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:import-new {--source= : Import from specific source} {--force : Force update even if version is same}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import new apps from multiple sources and update existing ones';

    /**
     * Execute the console command.
     */
    public function handle(ImportSourceManager $manager)
    {
        $source = $this->option('source');
        $force = $this->option('force');

        $this->info('Starting app import process...');
        $this->info('Source: ' . ($source ?: 'All sources'));
        $this->info('Force update: ' . ($force ? 'Yes' : 'No'));

        try {
            $sources = $manager->all();

            if (empty($sources)) {
                $this->error('No import sources configured.');
                return 1;
            }

            if ($source) {
                if (! $manager->has($source)) {
                    $this->error("Source '{$source}' not found.");
                    $this->info('Available sources: ' . implode(', ', array_keys($sources)));
                    return 1;
                }

                $this->processSource($manager, $source, $force);
            } else {
                foreach ($sources as $sourceKey => $sourceConfig) {
                    $this->processSource($manager, $sourceKey, $force);
                }
            }

            $this->info('Import process completed successfully!');
            return 0;
        } catch (Exception $e) {
            $this->error('Import process failed: ' . $e->getMessage());
            Log::error('App import failed', [
                'error' => $e->getMessage(),
                'source' => $source,
                'trace' => $e->getTraceAsString(),
            ]);
            return 1;
        }
    }

    /**
     * Process a single source feed.
     */
    protected function processSource(ImportSourceManager $manager, string $sourceKey, bool $force): void
    {
        $sourceConfig = $manager->config($sourceKey);
        $parser = $manager->parser($sourceKey);

        $this->info("Importing from {$sourceConfig['name']}...");

        $this->importCategories($sourceConfig, $parser->getCategories());
        $this->importApps($sourceConfig, $parser->getApps(), $force);

        $this->info("Successfully imported from {$sourceConfig['name']}");
    }

    /**
     * Import categories provided by the parser.
     */
    protected function importCategories(array $sourceConfig, array $categories): void
    {
        $this->line('Importing categories...');

        $importedCount = 0;

        foreach ($categories as $categoryData) {
            $slug = $categoryData['slug'] ?? $this->generateSlug($categoryData['name'] ?? 'category');

            $category = Category::updateOrCreate(
                ['slug' => $slug],
                [
                    'name' => $categoryData['name'] ?? ucfirst($slug),
                    'description' => $categoryData['description'] ?? null,
                    'icon' => $categoryData['icon'] ?? null,
                    'color' => $categoryData['color'] ?? ($sourceConfig['brand_color'] ?? null),
                    'status' => 'active',
                ]
            );

            if ($category->wasRecentlyCreated) {
                $importedCount++;
            }
        }

        $this->line("Imported {$importedCount} categories from {$sourceConfig['name']}");
    }

    /**
     * Import apps from the parsed feed.
     */
    protected function importApps(array $sourceConfig, array $apps, bool $force): void
    {
        $this->line('Importing apps...');

        if (empty($apps)) {
            $this->line('No apps were returned for this source.');
            return;
        }

        $importedCount = 0;
        $updatedCount = 0;
        $skippedCount = 0;

        $progressBar = $this->output->createProgressBar(count($apps));
        $progressBar->start();

        foreach ($apps as $appData) {
            try {
                $existingApp = App::where('package_name', $appData['package_name'])->first();

                $newVersion = isset($appData['version_code']) ? (int) $appData['version_code'] : null;

                $shouldUpdate = !$existingApp
                    || $force
                    || $this->isNewerVersion($newVersion, $existingApp->version_code ?? null);

                if (! $shouldUpdate && $existingApp) {
                    $skippedCount++;
                    $progressBar->advance();
                    continue;
                }

                $category = null;

                if (isset($appData['category']['slug'])) {
                    $category = Category::where('slug', $appData['category']['slug'])->first();
                }

                $appAttributes = $this->normalizeAppData($appData, $category, $existingApp?->version_code);

                $app = App::updateOrCreate(
                    ['package_name' => $appAttributes['package_name']],
                    $appAttributes
                );

                if ($app->wasRecentlyCreated) {
                    $importedCount++;
                } else {
                    $updatedCount++;
                    Log::info('App updated', [
                        'package_name' => $app->package_name,
                        'source' => $sourceConfig['name'],
                        'old_version' => $existingApp?->version_code ?? 'N/A',
                        'new_version' => $app->version_code,
                    ]);
                }

                if (!empty($appData['versions']) && is_array($appData['versions'])) {
                    $this->importVersions($app, $appData['versions']);
                }
            } catch (Exception $e) {
                $this->warn("Error processing app \"{$appData['name']}\": " . $e->getMessage());
                Log::error('Error processing app', [
                    'app_name' => $appData['name'] ?? 'Unknown',
                    'error' => $e->getMessage(),
                    'source' => $sourceConfig['name'],
                ]);
            }

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine();

        $this->line("Imported {$importedCount} new apps");
        $this->line("Updated {$updatedCount} existing apps");
        $this->line("Skipped {$skippedCount} apps (no updates needed)");
    }

    /**
     * Import versions if the source provides them.
     */
    protected function importVersions(App $app, array $versions): void
    {
        foreach ($versions as $versionData) {
            $app->versions()->updateOrCreate(
                ['version_code' => $versionData['code']],
                [
                    'version_name' => $versionData['name'],
                    'release_date' => isset($versionData['release_date']) ? Carbon::parse($versionData['release_date']) : null,
                    'changelog' => $versionData['changelog'] ?? null,
                    'download_url' => $versionData['download_url'] ?? null,
                    'size' => $versionData['size'] ?? null,
                    'min_os_version' => $versionData['min_os'] ?? null,
                    'max_os_version' => $versionData['max_os'] ?? null,
                    'is_beta' => $versionData['beta'] ?? false,
                    'status' => 'active',
                ]
            );
        }
    }

    /**
     * Prepare app data for persistence.
     */
    protected function normalizeAppData(array $data, ?Category $category, ?int $currentVersionCode = null): array
    {
        $latestVersion = $data['latest_version'] ?? $data['version'] ?? null;
        $versionCode = isset($data['version_code']) ? (int) $data['version_code'] : $currentVersionCode;

        return [
            'package_name' => $data['package_name'] ?? $this->generateSlug($data['name'] ?? 'app'),
            'slug' => $data['slug'] ?? $this->generateSlug($data['name'] ?? 'app'),
            'name' => $data['name'] ?? 'Unnamed App',
            'description' => $this->cleanDescription($data['description'] ?? ''),
            'icon_url' => $data['icon'] ?? $data['icon_url'] ?? null,
            'latest_version' => $latestVersion,
            'version_code' => $versionCode ?? 0,
            'rating' => isset($data['rating']) ? (float) $data['rating'] : 0,
            'downloads_count' => $data['downloads'] ?? 0,
            'os_requirement' => $data['os_requirement'] ?? null,
            'category_id' => $category?->id,
            'developer' => $data['developer'] ?? null,
            'size' => $data['size'] ?? null,
            'mod_info' => $data['mod_info'] ?? null,
            'screenshots' => $data['screenshots'] ?? [],
            'download_url' => $data['download_url'] ?? null,
            'status' => 'active',
            'featured' => $data['featured'] ?? false,
        ];
    }

    /**
     * Determine if the new version is newer than the current one.
     */
    protected function isNewerVersion(?int $newVersion, ?int $oldVersion): bool
    {
        return ($newVersion ?? 0) > ($oldVersion ?? 0);
    }

    /**
     * Generate a URL-friendly slug.
     */
    protected function generateSlug(string $name): string
    {
        $slug = strtolower($name);
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        return trim($slug, '-') ?: 'app';
    }

    /**
     * Strip and normalize description text.
     */
    protected function cleanDescription(string $description): string
    {
        $description = strip_tags($description);
        $description = preg_replace('/\s+/', ' ', $description);

        return trim($description);
    }
}