# Translation Guide for APK Center

## Overview

This project uses Laravel's built-in translation system with automatic translation support for **5 languages**:
- 🇬🇧 English (en)
- 🇸🇦 Arabic (ar)
- 🇪🇸 Spanish (es)
- 🇫🇷 French (fr)
- 🇮🇳 Hindi (hi)

## Quick Start: Auto-Translate Missing Strings

Run the automatic translation command:

```bash
# Make script executable
chmod +x translate.sh

# Run translations
./translate.sh
```

Or directly:

```bash
php artisan translate:strings --targets=ar,es,fr,hi
```

## Command Options

### Basic Usage
```bash
php artisan translate:strings
```

**Default behavior:**
- Source language: English (en)
- Target languages: Arabic, Spanish, French, Hindi
- Skips existing translations (doesn't overwrite)

### Custom Source Language
```bash
php artisan translate:strings --source=ar --targets=en,es,fr
```

### Force Retranslate All Strings
```bash
php artisan translate:strings --force
```
⚠️ This will retranslate and overwrite all existing translations.

### Custom Targets
```bash
php artisan translate:strings --targets=es,fr,de
```

## How It Works

1. **Reads** the source language file (default: `resources/lang/en.json`)
2. **Translates** each string using the free **MyMemory API** (no API key needed)
3. **Skips** strings already translated (unless `--force` flag is used)
4. **Saves** to the target language files
5. **Rate limits** to 100ms per request (respects API limits)

## Translation Files Location

All translation files are JSON format:

```
resources/lang/
├── en.json     (English - Source)
├── ar.json     (Arabic)
├── es.json     (Spanish)
├── fr.json     (French)
└── hi.json     (Hindi)
```

## Using Translations in Blade Templates

### Simple string translation
```blade
{{ __('Hello') }}
```

### Translation with parameters
```blade
{{ __('Welcome, :name', ['name' => $user->name]) }}
```

### Checking existence
```blade
@if(trans_has('message.key'))
    {{ __('message.key') }}
@endif
```

## Using Translations in PHP Code

```php
// In controllers or models
__('Some string')
trans('messages.key')
trans_choice('messages.items', $count)
```

## Adding New Translatable Strings

### 1. Add to English file first:
```json
{
    "Hello": "Hello",
    "Goodbye": "Goodbye"
}
```

### 2. Use in Blade:
```blade
{{ __('Hello') }}
```

### 3. Auto-translate to other languages:
```bash
./translate.sh
```

Or manually edit the JSON files with translations.

## Translation API Used

**MyMemory Translation API**
- Service: https://mymemory.translated.net
- Cost: **Completely Free**
- Rate Limit: ~500 requests/day (more than enough)
- Quality: Good for general phrases
- No API key required

### Fallback Strategy
If a translation fails:
1. Uses the source language string as fallback
2. Logs the warning
3. Continues processing other strings

## Best Practices

### ✅ DO:
- Use translation keys for all user-facing text
- Keep English keys clear and concise
- Group related translations logically
- Use parameter substitution for dynamic content

### ❌ DON'T:
- Hardcode text in Blade templates
- Use HTML/Markup in translation strings (translate text only)
- Leave untranslated strings in production

## Adding More Languages

To add a new language (e.g., German):

### 1. Create new JSON file:
```bash
touch resources/lang/de.json
```

### 2. Run auto-translation:
```bash
php artisan translate:strings --targets=de
```

### 3. Update app configuration:
```php
// config/app.php
'fallback_locale' => 'en',
'supported_locales' => ['en', 'ar', 'es', 'fr', 'hi', 'de'], // Add 'de'
```

## Troubleshooting

### Command not found
```bash
# Ensure artisan is executable
chmod +x artisan
```

### PHP not found
```bash
# Check PHP installation
php --version

# Or use full path
/usr/bin/php artisan translate:strings
```

### Translations not appearing
1. Check file permissions on `resources/lang/` directory
2. Clear cache: `php artisan cache:clear`
3. Verify locale is set correctly: `\App::getLocale()`

### API request timeout
- Network issue - try again later
- Reduce target languages in one batch
- Check internet connection

## Performance Optimization

The translation command includes:
- **Progress bar** for visual feedback
- **Rate limiting** (100ms between requests)
- **Alphabetical sorting** of keys
- **Batch processing** for efficiency

A typical run:
- 150 strings → ~15 seconds
- 300 strings → ~30 seconds
- Network dependent

## Advanced: Manual Translation

Edit JSON files directly:

```json
{
    "Hello": "مرحبا",
    "Goodbye": "وداعا",
    "Welcome": "أهلا وسهلا"
}
```

Keep format:
- UTF-8 encoding
- Valid JSON (use validator)
- Consistent indentation (4 spaces)
- Escaped quotes: `\"` 

## Support & Issues

If translations are incomplete or incorrect:

1. Check the language file exists
2. Review the JSON formatting
3. Re-run with `--force` flag
4. Manual edit for specific strings

## License & Attribution

- Translation powered by **MyMemory** (Free)
- Laravel **localization system**
- Your own custom command implementation

---

**Last Updated:** December 3, 2025
