<?php

/**
 * APK Scraper Routes - Add to routes/platform.php or routes/web.php
 *
 * This file shows how to register the APK import endpoints in your Laravel app.
 * Copy these routes into your routing configuration.
 *
 * @version 1.0
 * @date December 3, 2025
 */

// ============================================================================
// OPTION 1: API Routes (Recommended for Frontend Integration)
// ============================================================================
// Add to: routes/api.php

Route::middleware(['auth:sanctum'])->prefix('apk')->group(function () {
    use App\Http\Controllers\APKImportController;

    /**
     * Search and scrape apps
     * POST /api/apk/search
     *
     * Request:
     * {
     *   "source": "github|apkpure|apkmirror",
     *   "query": "search query or owner/repo"
     * }
     *
     * Response:
     * {
     *   "success": true,
     *   "total_found": 15,
     *   "valid_count": 12,
     *   "apps": [...]
     * }
     */
    Route::post('/search', [APKImportController::class, 'search'])
        ->name('api.apk.search')
        ->middleware(['auth:sanctum', 'admin']); // Require admin

    /**
     * Import a single app
     * POST /api/apk/import
     *
     * Request:
     * {
     *   "package_name": "com.example.app",
     *   "name": "Example App",
     *   "developer": "Developer Inc",
     *   ...
     * }
     *
     * Response: { "success": true, "app": {...} }
     */
    Route::post('/import', [APKImportController::class, 'import'])
        ->name('api.apk.import')
        ->middleware(['auth:sanctum', 'admin']);

    /**
     * Bulk import multiple apps
     * POST /api/apk/bulk-import
     *
     * Request:
     * {
     *   "apps": [{...}, {...}, ...]
     * }
     *
     * Response:
     * {
     *   "success": true,
     *   "imported": 5,
     *   "failed": 1,
     *   "results": [...]
     * }
     */
    Route::post('/bulk-import', [APKImportController::class, 'bulkImport'])
        ->name('api.apk.bulk-import')
        ->middleware(['auth:sanctum', 'admin']);

    /**
     * Get available sources and their capabilities
     * GET /api/apk/sources
     *
     * Response:
     * {
     *   "success": true,
     *   "sources": {
     *     "apkpure": {...},
     *     "apkmirror": {...},
     *     "github": {...}
     *   }
     * }
     */
    Route::get('/sources', [APKImportController::class, 'sources'])
        ->name('api.apk.sources');
});


// ============================================================================
// OPTION 2: Platform/Web Routes (For Orchid Admin Panel)
// ============================================================================
// Add to: routes/platform.php or routes/web.php

Route::middleware(['auth', 'admin'])->prefix('admin')->group(function () {
    use App\Http\Controllers\APKImportController;

    /**
     * Search endpoint for admin
     * POST /admin/apk-search
     */
    Route::post('/apk-search', [APKImportController::class, 'search'])
        ->name('admin.apk.search');

    /**
     * Import endpoint for admin
     * POST /admin/apk-import
     */
    Route::post('/apk-import', [APKImportController::class, 'import'])
        ->name('admin.apk.import');

    /**
     * Bulk import for admin
     * POST /admin/apk-bulk-import
     */
    Route::post('/apk-bulk-import', [APKImportController::class, 'bulkImport'])
        ->name('admin.apk.bulk-import');
});


// ============================================================================
// USAGE IN CONTROLLERS/VIEWS
// ============================================================================

/**
 * Example 1: Simple search in controller
 */
class SomeController extends Controller
{
    public function example1()
    {
        // Call via dispatch or queue
        $apps = MultiSourceScraper::scrape('github', 'mozilla-mobile/fenix');

        return view('apks.results', ['apps' => $apps]);
    }
}

/**
 * Example 2: JavaScript fetch
 */
?>

<script>
// Search with JavaScript
async function searchApps() {
    const source = document.getElementById('source').value;
    const query = document.getElementById('query').value;

    const response = await fetch('/api/apk/search', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
        },
        body: JSON.stringify({ source, query }),
    });

    const data = await response.json();

    if (data.success) {
        console.log(`Found ${data.valid_count} apps`);
        displayApps(data.apps);
    } else {
        console.error(data.message);
    }
}

// Import an app
async function importApp(appData) {
    const response = await fetch('/api/apk/import', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
        },
        body: JSON.stringify(appData),
    });

    const data = await response.json();

    if (data.success) {
        alert('App imported successfully!');
        // Reload or update UI
    } else {
        alert('Import failed: ' + JSON.stringify(data.errors));
    }
}

// Bulk import
async function bulkImportApps(appsList) {
    const response = await fetch('/api/apk/bulk-import', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
        },
        body: JSON.stringify({ apps: appsList }),
    });

    const data = await response.json();

    console.log(`Imported: ${data.imported}, Failed: ${data.failed}`);
    console.log(data.results);
}

// Get available sources
async function getSources() {
    const response = await fetch('/api/apk/sources');
    const data = await response.json();

    console.log('Available sources:', data.sources);
}
</script>

<?php

/**
 * Example 3: Blade template with forms
 */
?>

<!-- Search Form -->
<form method="POST" action="{{ route('admin.apk.search') }}">
    @csrf

    <select name="source" required>
        <option value="github">GitHub Releases</option>
        <option value="apkpure">APKPure</option>
        <option value="apkmirror">APKMirror</option>
    </select>

    <input type="text" name="query" placeholder="Search query or owner/repo" required>

    <button type="submit">Search & Scrape</button>
</form>

<!-- Results Table -->
@if($apps ?? false)
<table>
    <thead>
        <tr>
            <th>App Name</th>
            <th>Package</th>
            <th>Version</th>
            <th>Developer</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        @foreach($apps as $app)
        <tr>
            <td>{{ $app['name'] }}</td>
            <td>{{ $app['package_name'] }}</td>
            <td>{{ $app['latest_version'] }}</td>
            <td>{{ $app['developer'] }}</td>
            <td>
                <form method="POST" action="{{ route('admin.apk.import') }}" style="display:inline;">
                    @csrf
                    @foreach($app as $key => $value)
                        <input type="hidden" name="{{ $key }}" value="{{ $value }}">
                    @endforeach
                    <button type="submit">Publish</button>
                </form>
            </td>
        </tr>
        @endforeach
    </tbody>
</table>
@endif


// ============================================================================
// OPTIONAL: Middleware for Admin Check
// ============================================================================

<?php

/**
 * Create middleware: app/Http/Middleware/AdminOnly.php
 */

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class AdminOnly
{
    public function handle(Request $request, Closure $next)
    {
        // Check if user is admin
        if (!auth()->check() || !auth()->user()->is_admin) {
            abort(403, 'Unauthorized');
        }

        return $next($request);
    }
}

// Register in: app/Http/Kernel.php
protected $routeMiddleware = [
    // ...
    'admin' => \App\Http\Middleware\AdminOnly::class,
];


// ============================================================================
// EXAMPLE: Complete API Usage via cURL
// ============================================================================

/*
# Get sources
curl -X GET http://localhost:8000/api/apk/sources

# Search for apps
curl -X POST http://localhost:8000/api/apk/search \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer your_token" \
  -d '{
    "source": "github",
    "query": "mozilla-mobile/fenix"
  }'

# Import single app
curl -X POST http://localhost:8000/api/apk/import \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer your_token" \
  -d '{
    "package_name": "org.mozilla.fenix",
    "name": "Firefox",
    "developer": "Mozilla",
    "download_url": "https://...",
    "latest_version": "1.0",
    "source": "github"
  }'

# Bulk import
curl -X POST http://localhost:8000/api/apk/bulk-import \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer your_token" \
  -d '{
    "apps": [
      {...app1...},
      {...app2...}
    ]
  }'
*/


// ============================================================================
// SUMMARY
// ============================================================================

/*
ENDPOINTS:
- POST   /api/apk/search          → Search and scrape
- POST   /api/apk/import          → Import single app
- POST   /api/apk/bulk-import     → Import multiple apps
- GET    /api/apk/sources         → Get source info

MIDDLEWARE:
- auth:sanctum    → User must be authenticated
- admin           → User must be admin

RESPONSES:
- 200 OK          → Successful GET request
- 201 Created     → Successful POST (import)
- 207 Multi-Status → Bulk import with partial success
- 404 Not Found   → No apps found
- 422 Unprocessable Entity → Validation failed
- 500 Server Error → Server error

All endpoints return JSON:
{
  "success": boolean,
  "message": string,
  "data": object or array
}
*/
