# Quick Start — New Features Setup

Get the new features running in 10 minutes.

## Step 1: Register Routes

Add this to `routes/platform.php` (or create it if it doesn't exist):

```php
<?php

use App\Orchid\Screens\APKImportScreen;
use App\Orchid\Screens\ExportZipScreen;
use App\Http\Controllers\DownloadZipController;

Route::prefix('admin')->group(function () {
    // APK Import
    Route::get('apk-import', APKImportScreen::class)->name('platform.apk-import');
    Route::post('apk-import/scrape', [APKImportScreen::class, 'scrapeApps'])->name('platform.apk-import.scrape');
    Route::post('apk-import/publish', [APKImportScreen::class, 'publishApp'])->name('platform.apk-import.publish');
    Route::post('apk-import/bulk', [APKImportScreen::class, 'bulkPublish'])->name('platform.apk-import.bulk');

    // ZIP Export
    Route::get('export-zip', ExportZipScreen::class)->name('platform.export-zip');
    Route::post('export-zip/create', [ExportZipScreen::class, 'createArchive'])->name('platform.export-zip.create');
    Route::post('export-zip/delete', [ExportZipScreen::class, 'deleteArchive'])->name('platform.export-zip.delete');
    Route::get('download-zip/{file}', [DownloadZipController::class, 'download'])->name('platform.download-zip');
});
```

## Step 2: Create Download Controller

Create file: `app/Http/Controllers/DownloadZipController.php`

```php
<?php

namespace App\Http\Controllers;

use App\Services\ZipArchiveService;
use Illuminate\Support\Facades\Response;

class DownloadZipController extends Controller
{
    public function download(string $file)
    {
        $this->authorize('admin'); // Or your permission check
        
        $zip = new ZipArchiveService();
        $filePath = $zip->getArchivePath($file);

        if (!$filePath) {
            abort(404, 'Archive not found');
        }

        return Response::download($filePath, $file, [
            'Content-Type' => 'application/zip',
        ]);
    }
}
```

## Step 3: Add SEO to Your App View

In your app detail view (e.g., `resources/views/app/show.blade.php`):

```blade
@extends('layouts.app')

@section('content')
    {!! \App\Helpers\SEOHelper::renderAppMeta($app) !!}
    
    <!-- Rest of your view -->
    <div class="app-detail">
        <h1>{{ $app->name }}</h1>
        <p>{{ $app->description }}</p>
        <!-- ... -->
    </div>
@endsection
```

## Step 4: Generate Sitemap

Run once:
```bash
php artisan seo:generate-sitemap
```

Schedule daily in `app/Console/Kernel.php`:
```php
protected function schedule(Schedule $schedule)
{
    $schedule->command('seo:generate-sitemap')->daily();
}
```

## Step 5: Test New Features

### Test APK Import:
1. Visit admin panel: `/admin/apk-import` (if you added the route to platform menu)
2. Select "GitHub Releases" as source
3. Enter: `mozilla-mobile/fenix`
4. Click "Search & Scrape"
5. Click "Publish" on any result

### Test ZIP Export:
1. Visit admin panel: `/admin/export-zip`
2. Enter collection name: "Test Collection"
3. Check 2-3 apps
4. Click "Create ZIP Archive"
5. Click "Download"

### Test SEO:
1. Visit any app detail page
2. Open browser dev tools → Elements tab
3. Look for:
   - `<meta property="og:title">` (Open Graph)
   - `<script type="application/ld+json">` (Structured data)
   - `<link rel="canonical">` (Canonical URL)

## Step 6: Deploy

```bash
# Commit changes
git add -A
git commit -m "feat: add APK import, ZIP export, and SEO optimization"

# Deploy to your server
git push origin main
# or use your preferred deploy method
```

---

## Troubleshooting

### "Route not found" error
- Make sure routes are registered in the correct file
- Run: `php artisan route:list | grep apk`
- Restart Laravel queue/server if running

### Admin screens don't appear
- Check Platform menu is updated in `app/Orchid/PlatformProvider.php` ✅ (Done)
- Verify user has admin permission
- Clear cache: `php artisan config:clear`

### ZIP files not creating
- Check `storage/temp-zips` folder exists: `mkdir -p storage/temp-zips`
- Check write permissions: `chmod 755 storage/temp-zips`
- Check PHP ZipArchive extension: `php -m | grep zip`

### SEO meta tags not showing
- Make sure you added `{!! \App\Helpers\SEOHelper::renderAppMeta($app) !!}` to your Blade
- Check if `$app` variable is available in the view
- Use browser dev tools to inspect source

### GitHub scraper not finding releases
- Use format: `owner/repo` (e.g., `mozilla-mobile/fenix`)
- Check if repo is public
- API rate limit: 60 requests/hour unauthenticated

---

## Next Steps

- [ ] Add user authentication to import/export screens
- [ ] Implement real-time search (Meilisearch, Algolia)
- [ ] Add user ratings/reviews model
- [ ] Extend scraper for APKPure/APKMirror with Puppeteer
- [ ] Create frontend UI for favorites and collections
- [ ] Add email notifications for app updates
- [ ] Set up automated backups

---

**Questions?** Check `FEATURES_IMPLEMENTATION.md` for detailed documentation.